<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;
use App\Models\OrderItem;
use App\Models\OrderStatusHistory;

class OrderController extends Controller
{
   public function index()
{
    return view('admin.orders.list');
}



public function list(Request $request)
{
    $query = Order::with(['items.product','courier','shippingAddress'])
        ->select('orders.*');

    // DATE FILTER
    if ($request->from_date && $request->to_date) {
        $query->whereBetween('created_at', [
            $request->from_date,
            $request->to_date
        ]);
    }

    // STATUS FILTER
    if ($request->status) {
        $query->where('status',$request->status);
    }

    // CATEGORY FILTER
    if ($request->category_id) {
        $query->whereHas('items.product', function($q) use ($request){
            $q->where('category_id',$request->category_id);
        });
    }

    // SUBCATEGORY FILTER
    if ($request->subcategory_id) {
        $query->whereHas('items.product', function($q) use ($request){
            $q->where('subcategory_id',$request->subcategory_id);
        });
    }

    return DataTables::of($query)

        ->addIndexColumn()

        ->addColumn('customer', function($row){
            return optional($row->shippingAddress)->name ?? '-';
        })

        ->addColumn('items_count', function($row){
            return $row->items->count();
        })

        ->addColumn('delivery_number', function($row){
            return $row->tracking_id ?? '-';
        })

        ->editColumn('total_amount', function($row){
            return '₹ '.number_format($row->total_amount,2);
        })

        ->editColumn('payment_status', function($row){
            if($row->payment_status=='paid'){
                return '<span class="badge bg-success">Paid</span>';
            }
            return '<span class="badge bg-warning">Pending</span>';
        })

        ->editColumn('status', function($row){
            return '<span class="badge bg-info">'.$row->status.'</span>';
        })

        ->addColumn('action', function($row){
            return '
                <a href="'.route('orders.show',$row->id).'" class="btn btn-light btn-sm">View</a>
            ';
        })

        ->rawColumns(['payment_status','status','action'])

        ->make(true);
}
public function show($id)
{
    $order = Order::with([
        'items.product',
        'items.variant',
        'courier',
        'billingAddress',
        'shippingAddress',
        'statusHistory'
    ])->findOrFail($id);

    return view('admin.orders.show', compact('order'));
}

public function refundOrder(Request $request,$id)
{
    $order = Order::findOrFail($id);

    $order->status = 'refund';
 //   $order->payment_status = 'failed';
    $order->save();

    return response()->json([
        'success'=>true,
        'message'=>'Order Refunded Successfully'
    ]);
}


public function returnOrder(Request $request,$id)
{
    $order = Order::findOrFail($id);

    $order->status = 'returned';
    $order->save();

    return response()->json([
        'success'=>true,
        'message'=>'Order Returned Successfully'
    ]);
}


public function updateOrder(Request $request,$id)
{
    $order = Order::findOrFail($id);

    $order->status = $request->status ?? $order->status;
    $order->save();

    return response()->json([
        'success'=>true,
        'message'=>'Order Updated Successfully'
    ]);
}

public function updateStatus(Request $request,$id)
{
    $order = Order::findOrFail($id);

    $order->status = $request->status;
    $order->save();

    return response()->json([
        'success'=>true,
        'message'=>'Status Updated Successfully'
    ]);
}

}
